function [Dcompr] = dict_compr_ls(D, data, label, N, Tdata);
% dictionary atom selection based on Liu-Shah method

% D: init dictionary
% data: input signals used to obtain D
% label: class label for input signals
% N: number of dictionary atoms to be selected
% Tdata: sparsity
%
% Dcompr: compressed dictionary of N atoms
%

% Sam Q. Qiu (qiu@cs.umd.edu)
% Feb. 2011
%
% Reference:
%  Qiang Qiu, Zhuolin Jiang, and Rama Chellappa, "Sparse Dictionary-based
%  Representation and Recognition of Human Action Attributes", International
%  Conference on Computer Vision (ICCV) 2011
%
%  J. Liu and M. Shah. Learning human actions via information maximization, 2008. CVPR
%

ttt=cputime;

X = omp(D'*data,D'*D,Tdata);

N_dict=size(D,2); % number of dictionary items
N_class=length(unique(label)); % number of class
classid = unique(label);



p_y_d = zeros (N_dict, N_class); % P(Y|D)

for cid=1:N_class % class id
    idx = find(label==classid(cid));
    for did=1:N_dict % dictionary item id
        p_y_d(did, cid) = sum(abs(X(did, idx)))/length(idx)+eps;
    end
end

p_y_d = p_y_d./repmat(sum(p_y_d, 2), 1, size(p_y_d, 2));
p_d =sum(abs(X), 2)/ sum(sum(abs(X), 2)) +eps; % P(D)

minMI_set=[];

if N_dict==N
    Dcompr = D;
    Xcompr = X;
    return;
end

for iter=1:size(D, 2)-N
    minIM = inf;
    for d1=1:size(D, 2)-1
        
        E1 = p_d(d1)*sum(p_y_d(d1, :).*log2(p_y_d(d1, :)));
        
        for d2=d1+1:size(D, 2)
            E2 = p_d(d2)*sum(p_y_d(d2, :).*log2(p_y_d(d2, :)));
            
            p_d_tmp = p_d(d1)+p_d(d2);
            p_y_d_tmp = (p_d(d1)/p_d_tmp)*p_y_d(d1,:) + (p_d(d2)/p_d_tmp)*p_y_d(d2,:);
            
            E3 = p_d(d1)*sum(p_y_d(d1,:).*log2(p_y_d_tmp));
            E4 = p_d(d2)*sum(p_y_d(d2,:).*log2(p_y_d_tmp));
            
            MI=E1+E2-E3-E4;
            
            if MI < minIM
                minIM = MI;
                s = d1;
                t = d2;
            end
            
        end
    end
    
    d_merge = .5*D(:,s)+.5*D(:,t);
    p_d_merge = p_d(s)+p_d(t);
    p_y_d_merge = (p_d(s)/p_d_merge)*p_y_d(s,:) + (p_d(t)/p_d_merge)*p_y_d(t,:);
    
    D(:,[s t])=[];
    p_d([s t]) = [];
    p_y_d([s t],:) = [];
    
    D=[D d_merge];
    p_d = [p_d; p_d_merge];
    p_y_d = [p_y_d; p_y_d_merge];
    
end

D = D./repmat( sqrt(sum(D.^2,1)), size(D,1) ,1);
Dcompr = D;

ctime = cputime-ttt;
fprintf(1,'(LS) Dictionary compressed from %d to %d in %f sec\n',N_dict, N, ctime);