/**********************************************************/
/* This code is for PLDI-15 Artifact Evaluation only      */ 
/* and will be released with further copyright information*/ 
/* File: Sequential block w reexpansion of knapsack       */
/**********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include "getoptions.h"
#include <string.h>
#include <algorithm>
#include <iostream>
#include <fstream>

#include "harness.h"
#include "block.h"

#ifdef BLOCK_PROFILE
#include "blockprofiler.h"
BlockProfiler profiler(8);//set simd profiler simd width as 8
#endif

#ifdef TRACK_TRAVERSALS
uint64_t work = 0;
#endif

//int _expandDepth = 0;
long long _expandSize = D_MAX_BLOCK_SIZE;
int dynamic_reexpand_count = 0;
_Block * g_initial_block = NULL;
int g_is_partial = 0;

using namespace std;

/* every item in the knapsack has a weight and a value */
#define MAX_ITEMS 256

struct item {
  int value;
  int weight;
};


int best_so_far = INT_MIN;

int compare(struct item *a, struct item *b)
{
  double c = ((double) a->value / a->weight) -
      ((double) b->value / b->weight);

  if (c > 0)
    return -1;
  if (c < 0)
    return 1;
  return 0;
}

int read_input(const char *filename, struct item *items, int *capacity, int *n)
{
  int i;
  FILE *f;

  if (filename == NULL)
    filename = "\0";
  f = fopen(filename, "r");
  if (f == NULL) {
    fprintf(stderr, "open_input(\"%s\") failed\n", filename);
    return -1;
  }
  /* format of the input: #items capacity value1 weight1 ... */
  fscanf(f, "%d", n);
  fscanf(f, "%d", capacity);

  for (i = 0; i < *n; ++i)
    fscanf(f, "%d %d", &items[i].value, &items[i].weight);

  fclose(f);

  /* sort the items on decreasing order of value/weight */
  /* cilk2c is fascist in dealing with pointers, whence the ugly cast */
  qsort(items, *n, sizeof(struct item),
        (int (*)(const void *, const void *)) compare);

  return 0;
}

/*
 * return the optimal solution for n items (first is e) and
 * capacity c. Value so far is v.
 */
/*Pseudo tail recursive knapsack matching our language spec*/
void knapsack(struct item *e, int c, int n, int v, int* max_ret)
{
#ifdef TRACK_TRAVERSALS
  work++;
#endif
#ifdef BLOCK_PROFILE
  profiler.record_single();
#endif

  /* base case: full knapsack or no items */
  if (c < 0){
    *max_ret = max(*max_ret, INT_MIN);
    return;
  }

  if (n == 0 || c == 0){
    *max_ret = max(*max_ret, v);
    return;		/* feasible solution, with value v */
  }

  /*
   * compute the best solution without the current item in the knapsack
   */
  knapsack(e + 1, c, n - 1, v, max_ret);

  /* compute the best solution with the current item in the knapsack */
  knapsack(e + 1, c - e->weight, n - 1, v + e->value, max_ret);
}

int knapsack1(struct item * e, int n, _BlockStack* _stack, int* max_ret, int _depth);
int knapsack0(struct item * e, int n, _BlockStack* _stack, int* max_ret, int _depth);

/*Breadth First execution to expand the number of tasks in software block*/
void knapsack_expand_bf(struct item * e, int n, _BlockStack* _stack, int* max_ret, int* _depth){
  class _BlockSet* _set = _stack->get(*_depth);
  class _Block* _block = _set->block;
  class _Block* _nextBlock0 = &_set-> nextBlock0;
  _nextBlock0->recycle();
#ifdef BLOCK_PROFILE
  profiler.record(_block->size, *_depth);

  if (dynamic_reexpand_count) 
  {
    profiler.record_reexpansion(*_depth);
  }

  int c_bef_exp_size[2];
  int max_c_bef_exp_size = 0;
#endif   


  //Add left
  for (int _bi = 0; _bi < _block -> _Block::size; ++_bi) {
    class _Point &_point = _block ->  get (_bi);
    int c = _point.c0;
    int v = _point.v0;

    /* base case: full knapsack or no items */
    if (c < 0){
      *max_ret = max(*max_ret, INT_MIN);
      continue;
#ifdef PARALLELISM_PROFILE
      parallelismProfiler->recordTruncate();
#endif
    }

    if (n == 0 || c == 0){
      *max_ret = max(*max_ret, v);
      continue;		/* feasible solution, with value v */
#ifdef PARALLELISM_PROFILE
      parallelismProfiler->recordTruncate();
#endif
    }
    _nextBlock0->add(c, v, c - e->weight, v + e->value);
#ifdef PARALLELISM_PROFILE
    parallelismProfiler->recordRecurse();
#endif
  }

#ifdef BLOCK_PROFILE
  c_bef_exp_size[0] = _nextBlock0->size;
#endif
  //Add right
  for (int _bi = 0; _bi < _block -> _Block::size; ++_bi) {
    class _Point &_point = _block ->  get (_bi);
    int c = _point.c1;
    int v = _point.v1;

    /* base case: full knapsack or no items */
    if (c < 0){
      *max_ret = max(*max_ret, INT_MIN);
      continue;
#ifdef PARALLELISM_PROFILE
      parallelismProfiler->recordTruncate();
#endif
    }

    if (n == 0 || c == 0){
      *max_ret = max(*max_ret, v);
      continue;		/* feasible solution, with value v */
#ifdef PARALLELISM_PROFILE
      parallelismProfiler->recordTruncate();
#endif
    }

    _nextBlock0->add(c, v, c - e->weight, v + e->value);
#ifdef PARALLELISM_PROFILE
    parallelismProfiler->recordRecurse();
#endif
  }

#ifdef BLOCK_PROFILE
  c_bef_exp_size[1] =  _nextBlock0->size - c_bef_exp_size[0];
  max_c_bef_exp_size = max(c_bef_exp_size[0], c_bef_exp_size[1]);
#endif

  //Free old stack space
  if (!g_is_partial){
    if (!*_depth){
      delete g_initial_block;
    } else
    {
      _stack->release(*_depth - 1);
    }
  }

  int _nextblock0_size = _nextBlock0 -> _Block::size;
  *_depth += 1;
#ifdef BLOCK_PROFILE
  if (dynamic_reexpand_count == 0){
    if(_nextblock0_size) {
      profiler.record_bef_exp_size(*_depth, _nextblock0_size);
#ifdef INCLUSIVE
      profiler.record_w_wo_exp_ratio(*_depth, 1);	
#endif
    }
  } else{
    for (int i = 0; i < 2; ++i){
      if (c_bef_exp_size[i]) profiler.record_bef_exp_size(*_depth, c_bef_exp_size[i]);
    }
    if(_nextblock0_size) profiler.record_w_wo_exp_ratio(*_depth, ((double)_nextblock0_size) / max_c_bef_exp_size);	
  }
  if(_nextblock0_size) {
    profiler.record_aft_exp_size(*_depth, _nextblock0_size);
  }

#endif
  if (_nextblock0_size > 0 && _nextblock0_size <= _expandSize / 2) {
    _stack ->  get (*_depth) -> _BlockSet::block = _nextBlock0;
    knapsack_expand_bf(e + 1, n - 1, _stack, max_ret, _depth);
  } else { //Reach the buffer size, or finish all evaluation
    if (!dynamic_reexpand_count){// only print for the first time
      cout << "This is the max block buffer size for dfs: " << _nextblock0_size << endl;
    }
    if (_nextblock0_size){
      _stack ->  get (*_depth) -> _BlockSet::block = _nextBlock0;
      knapsack0(e + 1, n - 1, _stack, max_ret, *_depth);
      knapsack1(e + 1, n - 1, _stack, max_ret, *_depth);
    }

  }
#ifdef PARALLELISM_PROFILE
  parallelismProfiler->blockEnd();
#endif
}


/*Depth First execution of left children to limit the memory consumption*/
int knapsack0(struct item * e, int n, _BlockStack* _stack, int* max_ret, int _depth){
  class _BlockSet* _set = _stack->get(_depth);
  class _Block* _block = _set->block;
  class _Block* _nextBlock0 = &_set-> nextBlock0;
  _nextBlock0->recycle();

  int _block_size = _block->size;
  if (_block_size <= _expandSize / 2){//Do dynamic reexpand
    dynamic_reexpand_count++;
    g_is_partial = 1;
    knapsack_expand_bf(e, n, _stack, max_ret, &_depth);
    return 1;
  } else {
#ifdef BLOCK_PROFILE
    profiler.record(_block->size, _depth);
#endif
    for (int _bi = 0; _bi < _block -> _Block::size; ++_bi) {
      class _Point &_point = _block ->  get (_bi);
      int c = _point.c0;
      int v = _point.v0;

      /* base case: full knapsack or no items */
      if (c < 0){
        *max_ret = max(*max_ret, INT_MIN);
        continue;
#ifdef PARALLELISM_PROFILE
        parallelismProfiler->recordTruncate();
#endif
      }

      if (n == 0 || c == 0){
        *max_ret = max(*max_ret, v);
        continue;		/* feasible solution, with value v */
#ifdef PARALLELISM_PROFILE
        parallelismProfiler->recordTruncate();
#endif
      }

      _nextBlock0->add(c, v, c - e->weight, v + e->value);
#ifdef PARALLELISM_PROFILE
      parallelismProfiler->recordRecurse();
#endif
    }

    if (_nextBlock0 -> _Block::size > 0) {
      _stack ->  get (_depth + 1) -> _BlockSet::block = _nextBlock0;
#ifdef BLOCK_PROFILE
      profiler.record_bef_exp_size(_depth + 1, _nextBlock0->size);
      profiler.record_aft_exp_size(_depth + 1, _nextBlock0->size);
#ifdef INCLUSIVE
      profiler.record_w_wo_exp_ratio(_depth + 1, 1);	
#endif
#endif

      int skip = 0;
      skip = knapsack0(e + 1, n - 1, _stack, max_ret, _depth + 1);
      if (!skip)
        knapsack1(e + 1, n - 1, _stack, max_ret, _depth + 1);
    }
  }
#ifdef PARALLELISM_PROFILE
  parallelismProfiler->blockEnd();
#endif

  return 0;
}

/*Depth First execution of right children to limit the memory consumption*/
int knapsack1(struct item * e, int n, _BlockStack* _stack, int* max_ret, int _depth){
  class _BlockSet* _set = _stack->get(_depth);
  class _Block* _block = _set->block;
  class _Block* _nextBlock0 = &_set-> nextBlock0;
  _nextBlock0->recycle();

  int _block_size = _block->size;
  if (_block_size <= _expandSize / 2){//Do dynamic reexpand
    dynamic_reexpand_count++;
    g_is_partial = 1;
    knapsack_expand_bf(e, n, _stack, max_ret, &_depth);
    return 1;
  } else {
#ifdef BLOCK_PROFILE
    profiler.record(_block->size, _depth);
#endif
    for (int _bi = 0; _bi < _block -> _Block::size; ++_bi) {
      class _Point &_point = _block ->  get (_bi);
      int c = _point.c1;
      int v = _point.v1;

      /* base case: full knapsack or no items */
      if (c < 0){
        *max_ret = max(*max_ret, INT_MIN);
        continue;
#ifdef PARALLELISM_PROFILE
        parallelismProfiler->recordTruncate();
#endif
      }

      if (n == 0 || c == 0){
        *max_ret = max(*max_ret, v);
        continue;		/* feasible solution, with value v */
#ifdef PARALLELISM_PROFILE
        parallelismProfiler->recordTruncate();
#endif
      }

      _nextBlock0->add(c, v, c - e->weight, v + e->value);
#ifdef PARALLELISM_PROFILE
      parallelismProfiler->recordRecurse();
#endif
    }

    if (_nextBlock0 -> _Block::size > 0) {
      _stack ->  get (_depth + 1) -> _BlockSet::block = _nextBlock0;
#ifdef BLOCK_PROFILE
      profiler.record_bef_exp_size(_depth + 1, _nextBlock0->size);
      profiler.record_aft_exp_size(_depth + 1, _nextBlock0->size);
#ifdef INCLUSIVE
      profiler.record_w_wo_exp_ratio(_depth + 1, 1);	
#endif
#endif
      int skip = 0;
      skip = knapsack0(e + 1, n - 1, _stack, max_ret, _depth + 1);
      if (!skip)
        knapsack1(e + 1, n - 1, _stack, max_ret, _depth + 1);
    }
  }
#ifdef PARALLELISM_PROFILE
  parallelismProfiler->blockEnd();
#endif

  return 0;
}

int usage(void)
{
  fprintf(stderr, "\nUsage: knapsack [<cilk-options>] [-f filename] [-benchmark] [-h]\n\n");
  fprintf(stderr, "The 0-1-Knapsack is a standard combinatorial optimization problem: ``A\n");
  fprintf(stderr, "thief robbing a store finds n items; the ith item is worth v_i dollars\n");
  fprintf(stderr, "and weighs w_i pounds, where v_i and w_i are integers. He wants to take\n");
  fprintf(stderr, "as valuable a load as possible, but he can carry at most W pounds in\n");
  fprintf(stderr, "his knapsack for some integer W. What items should he take?''\n\n");
  return -1;
}

char *specifiers[] = {"-f", "-benchmark", "-h", "-b", 0};
int opt_types[] = {STRINGARG, BENCHMARK, BOOLARG, INTARG ,0};

/*Benchmark entrance called by harness*/
int app_main(int argc, char *argv[])
{
  struct item items[MAX_ITEMS];	/* array of items */
  int n, capacity, sol, benchmark, help, buf;
  char filename[100];
  buf = 0;
  sol = 0;

  /* standard benchmark options */
  strcpy(filename, "../inputs/knapsack-example2.input");

  get_options(argc, argv, specifiers, opt_types, filename, &benchmark, &help, &buf);

  if (buf) _expandSize = pow(2.0, buf);

  if (help)
    return usage();

  if (benchmark) {
    switch (benchmark) {
      case 1:		/* short benchmark options -- a little work */
        strcpy(filename, "../inputs/knapsack-example1.input");
        break;
      case 2:		/* standard benchmark options */
        strcpy(filename, "../inputs/knapsack-example2.input");
        break;
      case 3:		/* long benchmark options -- a lot of work */
        strcpy(filename, "../inputs/knapsack-example3.input");
        break;
    }
  }
  if (read_input(filename, items, &capacity, &n))
    return 1;

#ifdef PARALLELISM_PROFILE
  parallelismProfiler = new ParallelismProfiler;
#endif

  Harness::start_timing();
  //_expandDepth = Harness::get_splice_depth();

  //Initialize software block stack
  cout << "Set fixed max block buffer size, _expandSize: " << _expandSize << endl;
  _Block::max_block = _expandSize;
  Harness::set_block_size(_expandSize);
  class _BlockStack * _stack = new _BlockStack;
  class _Block * _block = new _Block;
  g_initial_block = _block;

  /* base case: full knapsack or no items */
  if (capacity < 0){
    sol = max(sol, INT_MIN);
    return 0;
  }

  if (n == 0 || capacity == 0){
    sol = max(sol, 0);
    return 0;		/* feasible solution, with value v */
  }

  _block->add(capacity, 0, capacity - items[0].weight, items[0].value);

  int _depth = 0;
  _stack->get (_depth) -> block = _block;

  //Start to execute blocked knapsack 
  if (_expandSize >= 2) {
#ifdef BLOCK_PROFILE
    profiler.record_bef_exp_size(0, 1);
    profiler.record_aft_exp_size(0, 1);
#ifdef INCLUSIVE
    profiler.record_w_wo_exp_ratio(0, 1);	
#endif
#endif
    knapsack_expand_bf(items + 1, n - 1, _stack, &sol, &_depth);
  }
  else{
    int df_block_size = _stack->get(_depth)->block->size;
    cout << "This is the max block buffer size for dfs: " << df_block_size << endl;

    if (df_block_size){
      knapsack0(items + _depth + 1, n - _depth - 1, _stack, &sol, _depth);
      knapsack1(items + _depth + 1, n - _depth - 1, _stack, &sol, _depth);
    }
  }


  delete _stack;
  if (_expandSize < 2) delete _block;

  Harness::stop_timing();
#ifdef BLOCK_PROFILE
  profiler.output();
#ifdef EXPAND_PROFILE
  profiler.outputReexpandInfo();//To show the reexpansion benefit, just comment out this line
#endif
#endif

#ifdef TRACK_TRAVERSALS
  cout << "work: " << work << endl;
#endif

  printf("\nExample: knapsack\n");
  printf("options: problem-file = %s\n\n", filename);
  printf("Best value is %d\n\n", sol);
  printf("This is dynamic reexpand counter: %d\n", dynamic_reexpand_count);

#ifdef PROFILE_SPACE_USE
  cout << "This is max space use (Bytes): " << m_space << endl;
  cout << "This is the total number of new operations for block: " << total_malloc << endl;
#endif

  return 0;
}
