/*
 * MS3DMaterial.java
 *
 * Copyright (C) 2001-2002 Kevin J. Duling (kevin@dark-horse.net)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.glyphein.j3d.loaders.milkshape;

import javax.vecmath.Color4f;

/** This object stores the material information for a MS3DGroup object.
 *
 * @author  Kevin J. Duling
 * @version $Revision: 1.3 $
 */
final class MS3DMaterial
{
  private String name;
  private Color4f ambient;
  private Color4f diffuse;
  private Color4f specular;
  private Color4f emissive;
  private float shininess;
  private float transparency;
  private byte mode;
  private String texture;
  private String alpha;

  /** Creates new MS3DMaterial */
  MS3DMaterial()
  {
  }
  
  /** Creates new MS3DMaterial.  The object is initialized by the provided parameters
   * @param name the name of the material
   * @param ambient the ambient color for the material
   * @param diffuse the diffuse color for the material
   * @param specular the specular color for the material
   * @param emissive the emissive color for the material
   * @param shininess the shininess of the material
   * @param transparency the transparency level of the material
   * @param mode unused
   * @param texture the filename of the primary texture
   * @param alpha  the filename of the secondary texture
   */
  MS3DMaterial(final String name,
               final Color4f ambient,
               final Color4f diffuse,
               final Color4f specular,
               final Color4f emissive,
               final float shininess,
               final float transparency,
               final byte mode,
               final String texture,
               final String alpha)
  {
    setName(name);
    setAmbient(ambient);
    setDiffuse(diffuse);
    setSpecular(specular);
    setEmissive(emissive);
    setShininess(shininess);
    setTransparency(transparency);
    setMode(mode);
    setTexture(texture);
    setAlphaMap(alpha);
  }
  
  /** Retrieve the name of this material
   * @return  a String
   */  
  final String getName()
  {
    return this.name;
  }
  
  /** Set the name of this material
   * @param name  a String
   */  
  final void setName(final String name)
  {
    this.name = name;
  }

  /** Retrieve the ambient color of this material
   * @return  a Color4f object
   */  
  final Color4f getAmbient()
  {
    return this.ambient;
  }
  
  /** Sets the ambient color of this material
   * @param ambient  a Color4f object
   */  
  final void setAmbient(final Color4f ambient)
  {
    this.ambient = ambient;
  }
  
  /** Retrieve the diffuse color of this material
   * @return  a Color4f object
   */  
  final Color4f getDiffuse()
  {
    return this.diffuse;
  }
  
  /** Sets the diffuse color of this material
   * @param diffuse  a Color4f object
   */  
  final void setDiffuse(final Color4f diffuse)
  {
    this.diffuse = diffuse;
  }
  
  /** Retrieve the specular color of this material
   * @return  a Color4f object
   */  
  final Color4f getSpecular()
  {
    return this.specular;
  }
  
  /** Sets the specular color of this material
   * @param specular  a Color4f object
   */  
  final void setSpecular(final Color4f specular)
  {
    this.specular = specular;
  }
  
  /** Retrieve the emissive color of this material
   * @return  a Color4f object
   */  
  final Color4f getEmissive()
  {
    return this.emissive;
  }
  
  /** Sets the emissive color of this material
   * @param emissive  a Color4f object
   */  
  final void setEmissive(final Color4f emissive)
  {
    this.emissive = emissive;
  }
  
  /** Retrieve the shininess value of this material
   * @return  a float
   */
  final float getShininess()
  {
    return this.shininess;
  }
  
  /** Set the shininess value of this material
   * @param shininess  a float
   */  
  final void setShininess(final float shininess)
  {
    this.shininess = shininess;
  }
  
  /** Retrieve the transparency value for this material
   * @return  a float
   */  
  final float getTransparency()
  {
    return this.transparency;
  }
  
  /** Set the transparency value for this material
   * @param transparency  a float
   */  
  final void setTransparency(final float transparency)
  {
    this.transparency = transparency;
  }
  
  /** not used
   * @return  a byte
   */  
  final byte getMode()
  {
    return this.mode;
  }
  
  /** not used
   * @param mode a byte
   */  
  final void setMode(final byte mode)
  {
    this.mode = mode;
  }
  
  /** Retrieve the filename of the primary texture for this material
   * @return  a String containing a filename
   */  
  final String getTexture()
  {
    return this.texture;
  }
  
  /** Set the filename of the primary texture for this material.
   * @param texture  a String containing a filename
   */  
  final void setTexture(final String texture)
  {
    this.texture = convertSlash(texture);
  }
  
  /** Retrieve the filename of the secondary texture for this material
   * @return  a String containing a filename
   */
  final String getAlphaMap()
  {
    return this.alpha;
  }
  
  /** Set the filename of the secondary texture for this material
   * @param alpha  a String containing a filename
   */  
  final void setAlphaMap(final String alpha)
  {
    this.alpha = convertSlash(alpha);
  }

  /** A utility function to convert Milkshape filenames to a platform-independent
   * format.  Milkshape stores all filenames in a DOS format.
   * @param filename the name of the file
   * @return the newly formatted string
   */  
  private static final String convertSlash(final String filename)
  {
    final int len = filename.length();
    char temp[] = new char[len];
    filename.getChars(0, len, temp, 0);
    for (int x = 0; x < len; x++)
      if (temp[x] == '\\')
        temp[x] = java.io.File.separatorChar;
    return new String(temp);
  }
}