/*
 * MS3DLoader.java
 *
 * Copyright (C) 2001-2002 Kevin J. Duling (kevin@dark-horse.net)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.glyphein.j3d.loaders.milkshape;

import java.awt.Color;
import java.awt.image.*;
import java.io.*;
import java.net.URL;
import java.util.ArrayList;
import javax.media.j3d.*;
import javax.vecmath.*;
import com.sun.j3d.loaders.*;
import com.sun.j3d.utils.image.TextureLoader;

import com.glyphein.util.LittleEndianDataInputStream;

/** Loads a Milkshape3D (version 1.4+) model.
 *
 * @author  Kevin J. Duling
 * @version $Revision: 1.12 $
 */
public final class MS3DLoader
  implements Loader
{
  private final int UNKNOWN = 0;
  private final int FROM_FILE = 1;
  private final int FROM_URL = 2;
  
  private final int LEN_NAME = 32;
  private final int LEN_FILENAME = 128;

  static private java.awt.Component observer;
  private LittleEndianDataInputStream ledis = null;
  private boolean bLoadFromFile = false;
  private boolean bLoadFromURL = false;
  private URL baseURL = null;
  private String basePath = null;
  private int flags = LOAD_ALL;
  private int source = UNKNOWN;

  /** Creates new MS3DLoader */
  public MS3DLoader()
  {
    this.flags = 0;
  }

  /** Creates new MS3DLoader.  Initializes flags with the provided variable.
   * Flags may be ORed to combine options.
   * @param flags a list of cabability flags.  Values may be ORed together.
   */
  public MS3DLoader(final int flags)
  {
    this.flags = flags;
  }

  /** DO NOT USE -- Not implemented.   This is only here for compatibility as the
   * general contract of the Loader interface requires implementation of the
   * load(java.io.Reader) method.
   * @param reader a java.io.Reader object
   * @throws FileNotFoundException Thrown if unable to locate the source file
   * @throws IncorrectFormatException Typically thrown if there is a problem loading the texture
   * @throws ParsingErrorException Unknown, included for compatibility
   * @return a Scene
   */
  public final Scene load(Reader reader)
    throws FileNotFoundException,
           IncorrectFormatException,
           ParsingErrorException
  {
    throw new UnsupportedOperationException("loading from java.io.Reader not supported");
  }
  
  /** This method loads the named file and returns the Scene containing the scene.
   * Any data files referenced by this file, such as textures, should be located in
   * the same place as the named file.  Alternatively, the model file
   * may contain a relative path for the texture within the texture name field.
   * For example, "./textures/marble.jpg"
   * @param str a String containing the fully qualified filename of the model to be loaded
   * @throws FileNotFoundException Thrown if unable to locate the source file
   * @throws IncorrectFormatException Typically thrown if there is a problem loading the texture
   * @throws ParsingErrorException Unknown, included for compatibility
   * @return a Scene
   */  
  public final Scene load(String str)
    throws FileNotFoundException,
           IncorrectFormatException,
           ParsingErrorException
  {
    source = FROM_FILE;
    if (this.getBasePath() != null)
      return loadModel(new FileInputStream(this.getBasePath() + str));
    else
      return loadModel(new FileInputStream(str));
  }

  /** This method loads the named url and returns the Scene containing the scene.
   * Any data files referenced by this file, such as textures, should be located in
   * the same place as the named file.  Alternatively, the model file
   * may contain a relative path for the texture within the texture name field.
   * For example, "./textures/marble.jpg"
   * @param uRL a URL object pointing to the model to be loaded
   * @throws FileNotFoundException Thrown if unable to locate the source file
   * @throws IncorrectFormatException Typically thrown if there is a problem loading the texture
   * @throws ParsingErrorException Unknown, included for compatibility
   * @return a Scene
   */  
  public final Scene load(URL uRL)
    throws FileNotFoundException,
           IncorrectFormatException, 
           ParsingErrorException
  {
    source = FROM_URL;
    setBaseUrl(uRL);
    try
    {
      return loadModel(uRL.openStream());
    }
    catch (IOException e)
    {
      System.err.println(e.getMessage());
      return null;
    }
  }
  
  /** Load the model from the input stream
   * @param is an InputStream object
   * @return the loaded Scene
   */
  private final Scene loadModel(final InputStream is)
  {
    final long mark = System.currentTimeMillis();
    final float[] texCoord = new float[2];
    final SceneBase scene = new SceneBase();
    final BranchGroup objGroup = new BranchGroup();
    boolean bAnim = false;
    try
    {
      ledis = new LittleEndianDataInputStream(is);
      final MS3DHeader header = loadHeader();
      final MS3DVertex vertList[] = loadVertexList();
      final MS3DTriangle triList[] = loadTriangleList();
      final MS3DGroup groupList[] = loadGroupList();
      final MS3DMaterial materialList[] = loadMaterialList();
      final float animationFPS = this.ledis.readFloat();
//System.out.println("Animation FPS: " + animationFPS);
      final float currentTime = this.ledis.readFloat();
//System.out.println("Current Time: " + currentTime);
      final int totalFrames = this.ledis.readInt();
//System.out.println("Total Frames: " + totalFrames);
      final MS3DJoint jointList[] = loadJointList();
      final int numJoints = jointList.length;
//System.out.println("Total Joints: " + numJoints);
      final int[] parentMap = buildParentJointMap(jointList, numJoints);

      final int numGroups = groupList.length;
      for (int i = 0; i < numGroups; i++)
      {
        Texture texture = null;
        Texture alphamap = null;
        int vertexCount = 0;
        MS3DGroup mesh = groupList[i];
        TriangleArray triangles = null;
        Shape3D shape = new Shape3D();
        Material mat = new Material();
        Appearance app = new Appearance();
        final int matIndex = mesh.getMaterialIndex();
        if (matIndex > -1)
        {
          final MS3DMaterial material = materialList[matIndex];

          Color4f color4 = material.getAmbient();
          mat.setAmbientColor(color4.x, color4.y, color4.z);

          color4 = material.getEmissive();
          mat.setEmissiveColor(color4.x, color4.y, color4.z);

          color4 = material.getSpecular();
          mat.setSpecularColor(color4.x, color4.y, color4.z);

          color4 = material.getDiffuse();
          mat.setDiffuseColor(color4.x, color4.y, color4.z);

          mat.setShininess(material.getShininess());

          final float trans = 1.0f - material.getTransparency();
          if (trans > 0.0f || material.getAlphaMap().length() > 0)
          {
            TransparencyAttributes transparency = new TransparencyAttributes();
            transparency.setTransparency(trans);
            transparency.setTransparencyMode(TransparencyAttributes.BLENDED);
            app.setTransparencyAttributes(transparency);
          }
          app.setMaterial(mat);

          texture = loadTexture(material.getTexture(), material.getAlphaMap());

          if (texture != null)
          {
            app.setTexture(texture);
            TextureAttributes ta = new TextureAttributes();
            ta.setTextureMode(TextureAttributes.MODULATE);
            ta.setTextureBlendColor(color4.x, color4.y, color4.z, 1.0f - color4.w); // diffuse color
            app.setTextureAttributes(ta);
          }
          if (texture != null)
            triangles = new TriangleArray(mesh.getTriangleCount() * 3, GeometryArray.COORDINATES |
                                                                       GeometryArray.NORMALS |
                                                                       GeometryArray.TEXTURE_COORDINATE_2);
        }
        if (triangles == null)
          triangles = new TriangleArray(mesh.getTriangleCount() * 3, GeometryArray.COORDINATES |
                                                                     GeometryArray.NORMALS);

        RenderingAttributes ra  = new RenderingAttributes();
        ra.setIgnoreVertexColors(true);
        app.setRenderingAttributes(ra);

        final int triangleCount = mesh.getTriangleCount();
        for (int j = 0; j < triangleCount; j++)
        {
          final int triangleIndex = mesh.getTriangleIndicies()[j];
          final MS3DTriangle triangle = triList[triangleIndex];
          final float s[] = triangle.getS();
          final float t[] = triangle.getT();
          for (int k = 0; k < 3; k++)
          {
            final int index = triangle.getVertexIndicies()[k];
            final Vector3f normals[] = triangle.getVertexNormals();
            final MS3DVertex vertex = vertList[index];
            if (!bAnim && vertex.getBoneID() != -1)
              bAnim = true;
            triangles.setCoordinate(vertexCount, vertex.getLocation());
            if (texture != null)
            {
              texCoord[0] = s[k];
              texCoord[1] = t[k];
              triangles.setTextureCoordinate(0, vertexCount, texCoord);
            }
            triangles.setNormal(vertexCount++, normals[k]);
          }
        }
        shape.setGeometry(triangles);
        shape.setAppearance(app);
        objGroup.addChild(shape);
        scene.addNamedObject(mesh.getName(), shape);
        if (bAnim)
        {
          shape.setCapability(Shape3D.ALLOW_GEOMETRY_READ);
          shape.setCapability(Shape3D.ALLOW_GEOMETRY_WRITE);
        }
      }
      Joint[] joints = setupJoints(jointList, vertList, triList);
    }
    catch (IOException e)
    {
      e.printStackTrace();
    }
    scene.setSceneGroup(objGroup);
//System.out.println("Total load time: " + (System.currentTimeMillis() - mark));
    return scene;
  }

  /** Loads the header of the file
   * @throws IOException thrown if an error occurs
   * @return  a MS3DHeader object
   */  
  private final MS3DHeader loadHeader()
    throws IOException
  {
    final byte id[] = new byte[10];
    this.ledis.read(id, 0, id.length);
    return new MS3DHeader(id, this.ledis.readInt());
  }

  /** Loads in the list of vertecies used in the model
   * @throws IOException thrown if an error occurs
   * @return  An array of MS3DVertex objects
   */
  private final MS3DVertex[] loadVertexList()
    throws IOException
  {
    final int numVerts = this.ledis.readUnsignedShort();
    final MS3DVertex vertList[] = new MS3DVertex[numVerts];
    for (int i = 0; i < numVerts; i++)
    {
      final byte flags = this.ledis.readByte();
      final Point3f vertex = new Point3f(this.ledis.readFloat(),
                                         this.ledis.readFloat(),
                                         this.ledis.readFloat());
      final byte boneID = this.ledis.readByte();
      final byte refCount = this.ledis.readByte();
      vertList[i] = new MS3DVertex(vertex, boneID, refCount, flags);
    }
    return vertList;
  }  

  /** Loads in all of the triangles (faces) of the object.  These are stored in an array
   * and looked up via index when building the Shape3D objects.
   * @throws IOException thrown if an error occurs
   * @return  an array of MS3DTriangle objects
   */
  private final MS3DTriangle[] loadTriangleList()
    throws IOException
  {
    final int numTris = this.ledis.readUnsignedShort();
    final MS3DTriangle triList[] = new MS3DTriangle[numTris];
    for (int x = 0; x < numTris; x++)
    {
      final MS3DTriangle triangle = new MS3DTriangle();
      triangle.setFlags(this.ledis.readUnsignedShort());

      final int vertIndicies[] = new int[3];
      vertIndicies[0] = this.ledis.readUnsignedShort();
      vertIndicies[1] = this.ledis.readUnsignedShort();
      vertIndicies[2] = this.ledis.readUnsignedShort();
      triangle.setVertexIndicies(vertIndicies);

      final Vector3f normals[] = new Vector3f[3];
      for (int i = 0; i < 3; i++)
        normals[i] = new Vector3f(this.ledis.readFloat(), this.ledis.readFloat(), this.ledis.readFloat());
      triangle.setVertexNormals(normals);

      final float s[] = new float[3];
      s[0] = this.ledis.readFloat();
      s[1] = this.ledis.readFloat();
      s[2] = this.ledis.readFloat();
      triangle.setS(s);

      final float t[] = new float[3];
      t[0] = 1.0f - this.ledis.readFloat();
      t[1] = 1.0f - this.ledis.readFloat();
      t[2] = 1.0f - this.ledis.readFloat();
      triangle.setT(t);

      triangle.setSmoothingGroup(this.ledis.readByte());
      triangle.setGroupIndex(this.ledis.readByte());
      triList[x] = triangle;
    }
    return triList;
  }

  /** Loads in the groups (meshes) in the model.
   * @throws IOException thrown if an error occurs
   * @return  an array of MS3DGroup objects
   */  
  private final MS3DGroup[] loadGroupList()
    throws IOException
  {
    final int numGroups = this.ledis.readUnsignedShort();
    final MS3DGroup groupList[] = new MS3DGroup[numGroups];
    for (int x = 0; x < numGroups; x++)
    {
      final MS3DGroup group = new MS3DGroup();
      group.setFlags(this.ledis.readByte());
      final byte name[] = new byte[this.LEN_NAME];
      this.ledis.read(name, 0, name.length);
      group.setName(makeSafeString(name));
      final int indexCount = this.ledis.readUnsignedShort();
      final int indicies[] = new int[indexCount];
      for (int i = 0; i < indexCount; i++)
        indicies[i] = this.ledis.readUnsignedShort();
      group.setTriangleIndicies(indicies);
      group.setMaterialIndex(this.ledis.readByte());
      groupList[x] = group;
    }
    return groupList;
  }

  /** Loads in the materials used in the model
   * @throws IOException thrown if there is a problem reading the model
   * @return an array of MS3DMaterial objects
   */
  private final MS3DMaterial[] loadMaterialList()
    throws IOException
  {
    final int numMaterials = this.ledis.readUnsignedShort();
    final MS3DMaterial materialList[] = new MS3DMaterial[numMaterials];
    for (int x = 0; x < numMaterials; x++)
    {
      final MS3DMaterial mat = new MS3DMaterial();
      final byte name[] = new byte[this.LEN_NAME];
      this.ledis.read(name, 0, name.length);
      mat.setName(makeSafeString(name));
      mat.setAmbient(new Color4f(this.ledis.readFloat(), this.ledis.readFloat(),
                                 this.ledis.readFloat(), this.ledis.readFloat()));
      mat.setDiffuse(new Color4f(this.ledis.readFloat(), this.ledis.readFloat(),
                                 this.ledis.readFloat(), this.ledis.readFloat()));
      mat.setSpecular(new Color4f(this.ledis.readFloat(), this.ledis.readFloat(),
                                  this.ledis.readFloat(), this.ledis.readFloat()));
      mat.setEmissive(new Color4f(this.ledis.readFloat(), this.ledis.readFloat(),
                                  this.ledis.readFloat(), this.ledis.readFloat()));
      mat.setShininess(this.ledis.readFloat());
      mat.setTransparency(this.ledis.readFloat());
      mat.setMode(this.ledis.readByte());
      final byte texture[] = new byte[this.LEN_FILENAME];
      this.ledis.read(texture, 0, texture.length);
      mat.setTexture(makeSafeString(texture));
      final byte alphamap[] = new byte[this.LEN_FILENAME];
      this.ledis.read(alphamap, 0, alphamap.length);
      mat.setAlphaMap(makeSafeString(alphamap));

      materialList[x] = mat;
    }
    return materialList;
  }

  /** Loads in the joints used in animating the model
   * @throws IOException thrown if an error occurs
   * @return  an array of MS3DJoint objects
   */
  private final MS3DJoint[] loadJointList()
    throws IOException
  {
    final int numJoints = this.ledis.readUnsignedShort();
    final MS3DJoint jointList[] = new MS3DJoint[numJoints];
    for (int x = 0; x < numJoints; x++)
    {
      final MS3DJoint joint = new MS3DJoint();
      joint.setFlags(this.ledis.readByte());
      final byte name[] = new byte[this.LEN_NAME];
      this.ledis.read(name, 0, name.length);
      joint.setName(makeSafeString(name));
      
      final byte parent[] = new byte[this.LEN_NAME];
      this.ledis.read(parent, 0, parent.length);
      joint.setParentName(makeSafeString(parent));
      
      joint.setRotation(new Vector3f(this.ledis.readFloat(), this.ledis.readFloat(), this.ledis.readFloat()));
      joint.setPosition(new Vector3f(this.ledis.readFloat(), this.ledis.readFloat(), this.ledis.readFloat()));

      final int numKeyFramesRot = this.ledis.readUnsignedShort();
      final int numKeyFramesPos = this.ledis.readUnsignedShort();
      final MS3DKeyFrameRotation keyFramesRot[] = new MS3DKeyFrameRotation[numKeyFramesRot];
      final MS3DKeyFramePosition keyFramesPos[] = new MS3DKeyFramePosition[numKeyFramesPos];

      for (int i = 0; i < numKeyFramesRot; i++)
      {
        final MS3DKeyFrameRotation keyFrameRot = new MS3DKeyFrameRotation();
        keyFrameRot.setTime(this.ledis.readFloat());
        keyFrameRot.setRotation(new Vector3f(this.ledis.readFloat(), this.ledis.readFloat(), this.ledis.readFloat()));
        keyFramesRot[i] = keyFrameRot;
      }
      joint.setRotationKeyFrames(keyFramesRot);

      for (int i = 0; i < numKeyFramesPos; i++)
      {
        final MS3DKeyFramePosition keyFramePos = new MS3DKeyFramePosition();
        keyFramePos.setTime(this.ledis.readFloat());
        keyFramePos.setPosition(new Point3f(this.ledis.readFloat(), this.ledis.readFloat(), this.ledis.readFloat()));
        keyFramesPos[i] = keyFramePos;
      }
      joint.setPositionKeyFrames(keyFramesPos);

      jointList[x] = joint;
    }
    return jointList;
  }

  /** Converts an array of bytes into a java.lang.String object.  The String is created by
   * searching for the first occurrence of null ('\0').  If a null is not within the byte
   * array, the entire array is used to create the String.
   * @param buffer - an array of bytes
   * @return  a String
   */
  private final String makeSafeString(final byte buffer[])
  {
    final int len = buffer.length;
    for (int i = 0; i < len; i++)
      if (buffer[i] == (byte)0)
        return new String(buffer, 0, i);
    return new String(buffer);
  }

  /** Returns the current loading flags setting.
   * @return  an integer value
   */
  public final int getFlags()
  {
    return this.flags;
  }
  
  /** Returns the current base path setting.  By default this is null, implying
   * the loader should look for associated files starting from the same directory
   * as the file passed into the load(String) method.
   * @return  a String
   */  
  public final String getBasePath()
  {
    return this.basePath;
  }
  
  /** This method sets the load flags for the file.  The flags should equal 0 by
   * default (which tells the loader to only load geometry).  To enable the loading
   * of any particular scene elements, pass in a logical OR of the LOAD values
   * specified above.
   * @param flags  an integer value
   */  
  public final void setFlags(final int flags)
  {
    this.flags = flags;
  }
  
  /** Returns the current base URL setting.  By default this is null, implying
   * the loader should loko for associated files starting from the same directory
   * as the file passed into the load(URL) method.
   * @return a java.net.URL object
   */  
  public final URL getBaseUrl()
  {
    return this.baseURL;
  }
  
  /** This method sets the base path name for data files associated with the file
   * passed into the load(String) method.  The basePath should be null by default,
   * which is an indicator to the loader that it should look for any associated
   * files starting fromt he same directory as the file passed into the load(String)
   * method.
   * @param str  a pathname
   */  
  public final void setBasePath(final String str)
  {
    this.basePath = str;
  }
  
  /** This method sets the base URL name for data files associated with the file
   * passed into the load(URL) method.  The basePath should be null by default,
   * which is an indicator to the loader that it should look for any associated
   * files starting from the same directory as the file passed into the load(URL)
   * method.
   * @param uRL  a java.net.URL object
   */
  public final void setBaseUrl(final URL uRL)
  {
    this.baseURL = uRL;
  }
  
  /** Loads the texture from the provided file.  If an alphamap is provided, then the
   * alpha values will be automatically calculated.  The formula to generate an alphamap
   * from RGB values is A = (R + G + B) / 3.
   * @param texFile the texture filename
   * @param alphaFile the alphamap file name
   * @throws IOException thrown if there is a problem reading the files
   * @return a Texture object
   */  
  private final Texture loadTexture(String texFile, String alphaFile)
    throws IOException
  {
    Texture texture = null;
    TextureLoader texLoader = null;
    if (texFile.length() > 0)
    {
      switch (source)
      {
        case FROM_FILE:
          if (this.getBasePath() != null)
            texFile = this.getBasePath() + texFile;
          texLoader = new TextureLoader(texFile, observer);
          texture = texLoader.getTexture();
          break;
        case FROM_URL:
          texLoader = new TextureLoader(new URL(getBaseUrl(), urlSlash(texFile)), observer);
          texture = texLoader.getTexture();
          break;
      }
    }

    // if we're going to use an alphamap, we need to build a new texture
    if (texture != null && alphaFile.length() > 0)
    {
      TextureLoader alphaLoader = null;
      switch (source)
      {
        case FROM_FILE:
          if (this.getBasePath() != null)
            alphaFile = this.getBasePath() + alphaFile;
          alphaLoader = new TextureLoader(alphaFile, observer);
          break;
        case FROM_URL:
          alphaLoader = new TextureLoader(new URL(getBaseUrl(), urlSlash(alphaFile)), observer);
          break;
      }
      final int width = texture.getWidth();
      final int height = texture.getHeight();
      final int length = width * height;

      final BufferedImage buffer = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
      final BufferedImage alphaBuff = alphaLoader.getScaledImage(width, height).getImage();
      final BufferedImage texBuff = texLoader.getImage().getImage();
      
      final int temp[] = new int[length];
      final int rgb[] = new int[length];
      alphaBuff.getRGB(0, 0, width, height, temp, 0, width);
      texBuff.getRGB(0, 0, width, height, rgb, 0, width);
      int x = -1;
      while (++x < length)
      {
        final int a = ((((temp[x] & 0xff0000) >> 16) + ((temp[x] & 0xff00) >> 8) + (temp[x] & 0xff)) / 3);
        temp[x] = (a << 24) + (rgb[x] & 0xffffff);
      }
      buffer.setRGB(0, 0, width, height, temp, 0, width);

      texture = new Texture2D(Texture2D.BASE_LEVEL, Texture2D.RGBA, width, height);
      texture.setImage(0, new ImageComponent2D(ImageComponent2D.FORMAT_RGBA, buffer));
    }
    
    return texture;
  }
  
  /** Replaces all instances of the file separator character with a forward slash ('/').
   * This function is useful when handling URL resources.  Milkshape is a Win32-only 
   * program and therefore stores the DOS path separator character ('\') in the
   * texture filename (e.g., "..\textures\brick.jpg").  When the texture filename is loaded
   * in MS3DMaterial, conversion from the DOS backslash to java.io.File.separatorChar is
   * automatically done.  It's only in the case of URL resources where more work needs to
   * be done.  URLs will always use a forward slash ('/') as their separator, 
   * the webserver converting to the native separator internally.
   * @param filename The filename to parse
   * @return the reformatted string
   * @see MS3DMaterial
   */  
  private static final String urlSlash(final String filename)
  {
    final int len = filename.length();
    char temp[] = new char[len];
    filename.getChars(0, len, temp, 0);
    for (int x = 0; x < len; x++)
      if (temp[x] == java.io.File.separatorChar)
        temp[x] = '/';
    return new String(temp);
  }

  private final int[] buildParentJointMap(final MS3DJoint[] jointList, final int numJoints)
  {
    final int[] parentMap = new int[numJoints];
    for (int x = 0; x < numJoints; x++)
    {
      if (jointList[x].getParentName().length() > 0)
        for (int y = 0; y < numJoints; y++)
        {
          if (jointList[x].getParentName().equals(jointList[y].getName()))
          {
            parentMap[x] = y;
            break;
          }
          parentMap[x] = -1;
        }
      else
        parentMap[x] = -1;
    }
    return parentMap;
  }
  
  private final Joint[] setupJoints(final MS3DJoint[] jointList,
                                    final MS3DVertex[] vertList,
                                    final MS3DTriangle[] triList)
  {
    final int numJoints = jointList.length;
    final int numVerts = vertList.length;
    final int numTris = triList.length;

    Joint[] joints = new Joint[numJoints];

    for (int x = 0; x < numJoints; x++)
    {
      Joint joint = new Joint();
      joint.id = x;
      if (jointList[x].getParentName().length() > 0)
        for (int y = 0; y < numJoints; y++)
        {
          if (jointList[x].getParentName().equals(jointList[y].getName()))
          {
            joint.parentId = y;
            break;
          }
          joint.parentId = -1;
        }
      else
        joint.parentId = -1;
      joint.localRotation = jointList[x].getRotation();
      joint.localTranslation = jointList[x].getPosition();
      joint.relative = setRotationRadians(joint.localRotation);
      joint.relative.setTranslation(joint.localTranslation);
      if (joint.parentId == -1)
        joint.absolute.set(joint.relative);
      else
      {
        joint.absolute.set(joints[joint.parentId].absolute);
        joint.absolute.mul(joint.relative);
      }
      joints[x] = joint;
    }
    
    for (int x = 0; x < numVerts; x++)
    {
      MS3DVertex vertex = vertList[x];
      if (vertex.getBoneID() != -1)
      {
        final Matrix4f matrix = joints[vertex.getBoneID()].absolute;
        inverseTranslateVect(vertex.getLocation());
        inverseRotateVect(vertex.getLocation());
      }
    }
    
    for (int x = 0; x < numTris; x++)
    {
      MS3DTriangle triangle = triList[x];
      for (int y = 0; y < 3; y++)
      {
        final MS3DVertex vertex = vertList[triangle.getVertexIndicies()[y]];
        if (vertex.getBoneID() != -1)
        {
          final Matrix4f matrix = joints[vertex.getBoneID()].absolute;
          inverseRotateVect(triangle.getVertexNormals()[y]);
        }
      }
    }
    
    
    return joints;
  }

  private final static Matrix4f setRotationRadians(final Vector3f angles)
  {
    Matrix4f ret = new Matrix4f();
    ret.setIdentity();
    final double cr = Math.cos(angles.x);
    final double sr = Math.sin(angles.x);
    final double cp = Math.cos(angles.y);
    final double sp = Math.sin(angles.y);
    final double cy = Math.cos(angles.z);
    final double sy = Math.sin(angles.z);
    final double srsp = sr*sp;
    final double crsp = cr*sp;
    
    ret.m00 = (float)(cp*cy);
    ret.m10 = (float)(cp*sy);
    ret.m20 = (float)(-sp);

    ret.m01 = (float)(srsp*cy - cr*sy);
    ret.m11 = (float)(srsp*sy + cr*cy);
    ret.m21 = (float)(sr*cp);
    
    ret.m02 = (float)(crsp*cy + sr*sy);
    ret.m12 = (float)(crsp*sy - sr*cy);
    ret.m22 = (float)(cr*cp);
    return ret;
  }
  
  private final static Matrix4f inverseRotateVect(final Tuple3f vector)
  {
    return new Matrix4f();
  }

  private final static Matrix4f inverseTranslateVect(final Tuple3f vector)
  {
    return new Matrix4f();
  }
}
