/*
 * MS3DJoint.java
 *
 * Copyright (C) 2001-2002 Kevin J. Duling (kevin@dark-horse.net)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.glyphein.j3d.loaders.milkshape;

import javax.vecmath.Vector3f;

/** MS3DJoint contains joint information, used for animating a Milkshape3D model.
 *
 * @author  Kevin J. Duling
 * @version $Revision: 1.3 $
 */
final class MS3DJoint
{
  private byte flags;
  private String name;
  private String parent;
  private Vector3f rotation;
  private Vector3f position;
  private short numKeyFramesRot;
  private short numKeyFramesTrans;
  private MS3DKeyFrameRotation keyFramesRot[];
  private MS3DKeyFramePosition keyFramesPos[];

  /** Creates new MS3DJoint */
  MS3DJoint()
  {
  }
  
  /** Creates a new MS3DJoint.  Values are initialized by the provided parameters.
   * @param name the name of the joint
   * @param parent the name of the parent joint
   * @param rotation the rotaiton of the joint
   * @param position the position of the joint
   * @param keyFramesRot the keyframes for rotations
   * @param keyFramesPos  the keyframes for positions
   */  
  MS3DJoint(final String name,
            final String parent,
            final Vector3f rotation,
            final Vector3f position,
            final MS3DKeyFrameRotation keyFramesRot[],
            final MS3DKeyFramePosition keyFramesPos[])
  {
    setName(name);
    setParentName(parent);
    setRotation(rotation);
    setPosition(position);
    setRotationKeyFrames(keyFramesRot);
    setPositionKeyFrames(keyFramesPos);
  }
  
  /** Retreive the name of this joint
   * @return  a String
   */  
  final String getName()
  {
    return this.name;
  }
  
  /** Sets the name of this joint.
   * @param name  a String
   */  
  final void setName(final String name)
  {
    if (name.length() > 32)
      throw new IllegalArgumentException("MS3DJoint: name is " + name.length() + ", can only be 32");
    this.name = name;
  }
  
  /** Retrieve the name of the parent joint.
   * @return  a String
   */  
  final String getParentName()
  {
    if (name.length() > 32)
      throw new IllegalArgumentException("MS3DJoint: parent name is " + name.length() + ", can only be 32");
    return this.parent;
  }
  
  /** Tell this joint what the name of its parent is.
   * @param name  a String
   */  
  final void setParentName(final String name)
  {
    this.parent = name;
  }

  /** Retrieve the flags set on this joint.
   * @return  a byte
   */  
  final byte getFlags()
  {
    return this.flags;
  }
  
  /** Set the flags on this joint.
   * @param flags  a byte
   */  
  final void setFlags(final byte flags)
  {
    this.flags = flags;
  }
  
  /** Retrieve the position of this joint.
   * @return  a Vector3f representing the position
   */  
  final Vector3f getPosition()
  {
    return this.position;
  }
  
  /** Set the position of this joint.
   * @param position  a Vector3f representing this joint's position
   */  
  final void setPosition(final Vector3f position)
  {
    this.position = position;
  }
  
  /** Retrieve the rotation of this joint.
   * @return  a Vector3f representing the rotation
   */  
  final Vector3f getRotation()
  {
    return this.rotation;
  }
  
  /** Set the rotation of this joint.
   * @param rotation  a Vector3f representing this joint's rotation
   */  
  final void setRotation(final Vector3f rotation)
  {
    this.rotation = rotation;
  }
  
  /** Retrieve the number of rotational key frames.
   * @return  a short
   */  
  final short getRotationKeyFramesCount()
  {
    return (short)this.keyFramesRot.length;
  }

  /** Retrieve the number of positional key frames.
   * @return  a short
   */  
  final short getPositionKeyFramesCount()
  {
    return (short)this.keyFramesPos.length;
  }
  
  /** Retreive the Rotation Key Frames
   * @return  an array of MS3DKeyFrameRotation objects
   */  
  final MS3DKeyFrameRotation[] getRotationKeyFrames()
  {
    return this.keyFramesRot;
  }

  /** Set the rotation key frames
   * @param frames  an array of MS3DKeyFrameRotation objects */  
  final void setRotationKeyFrames(final MS3DKeyFrameRotation frames[])
  {
    this.keyFramesRot = frames;
  }

  /** Retreive the position key frames
   * @return  an array of MS3DKeyFramePosition objects
   */  
  final MS3DKeyFramePosition[] getPositionKeyFrames()
  {
    return this.keyFramesPos;
  }

  /** Set the position key frames
   * @param frames  an array of setPositionKeyFrames objects */  
  final void setPositionKeyFrames(final MS3DKeyFramePosition frames[])
  {
    this.keyFramesPos = frames;
  }
}